<?php
require_once '../config.php';
require_once '../functions.php';
requireLogin();

$analysisData = [];
$chartData = [];

if (checkDataFile()) {
    $data = readCSVData();
    $engineerData = [];
    
    // Process all rows
    foreach ($data as $row) {
        $engName = $row[16] ?? 'Unknown';
        $woNo = $row[10] ?? '';
        $allocatedHours = floatval($row[21] ?? 0);
        $workTime = floatval($row[24] ?? 0);
        
        if (!isset($engineerData[$engName])) {
            $engineerData[$engName] = [
                'jobs' => [],
                'total_allocated' => 0,
                'total_actual' => 0,
                'exceeded_jobs' => 0
            ];
        }
        
        if (!in_array($woNo, $engineerData[$engName]['jobs'])) {
            $engineerData[$engName]['jobs'][] = $woNo;
        }
        
        $engineerData[$engName]['total_allocated'] += $allocatedHours;
        $engineerData[$engName]['total_actual'] += $workTime;
        
        if ($workTime > $allocatedHours) {
            $engineerData[$engName]['exceeded_jobs']++;
        }
    }
    
    // Calculate final metrics
    foreach ($engineerData as $name => $data) {
        $jobCount = count($data['jobs']);
        $avgAllocated = $jobCount > 0 ? $data['total_allocated'] / $jobCount : 0;
        $avgActual = $jobCount > 0 ? $data['total_actual'] / $jobCount : 0;
        $exceededAny = $data['exceeded_jobs'] > 0 ? 'Yes' : 'No';
        $overtime = max(0, $data['total_actual'] - $data['total_allocated']);
        $overtimeRatio = $data['total_allocated'] > 0 ? $overtime / $data['total_allocated'] : 0;
        
        $analysisData[] = [
            'engineer' => $name,
            'jobs' => $jobCount,
            'total_allocated' => $data['total_allocated'],
            'total_actual' => $data['total_actual'],
            'avg_allocated' => $avgAllocated,
            'avg_actual' => $avgActual,
            'exceeded_jobs' => $data['exceeded_jobs'],
            'exceeded_any' => $exceededAny,
            'overtime_hours' => $overtime,
            'overtime_ratio' => $overtimeRatio
        ];
    }
    
    // Sort by overtime hours descending
    usort($analysisData, function($a, $b) {
        return $b['overtime_hours'] <=> $a['overtime_hours'];
    });
    
    // Prepare chart data (top 10 for overtime)
    $top10 = array_slice($analysisData, 0, 10);
    foreach ($top10 as $item) {
        $chartData['engineers'][] = $item['engineer'];
        $chartData['overtime'][] = round($item['overtime_hours'], 2);
        $chartData['allocated'][] = round($item['total_allocated'], 2);
        $chartData['actual'][] = round($item['total_actual'], 2);
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Engineer Performance - <?php echo APP_NAME; ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Manrope', sans-serif; background: #f8fafc; color: #1e293b; }
        .container { display: flex; min-height: 100vh; }
        
        .sidebar { width: 260px; background: white; border-right: 1px solid #e2e8f0; padding: 24px 0; position: fixed; height: 100vh; overflow-y: auto; }
        .sidebar-header { padding: 0 24px 24px; border-bottom: 1px solid #e2e8f0; display: flex; align-items: center; gap: 12px; }
        .logo { height: 32px; }
        .app-name { font-size: 20px; font-weight: 700; color: #0066b2; }
        .menu { list-style: none; padding: 16px 0; }
        .menu li { margin-bottom: 4px; }
        .menu a { display: flex; align-items: center; gap: 12px; padding: 12px 24px; color: #64748b; text-decoration: none; font-size: 15px; font-weight: 500; transition: all 0.2s; }
        .menu a:hover, .menu a.active { background: #f1f5f9; color: #0066b2; }
        .menu-icon { width: 20px; height: 20px; display: flex; align-items: center; justify-content: center; }
        
        .main-content { flex: 1; margin-left: 260px; padding: 32px; }
        .topbar { background: white; border-radius: 12px; padding: 16px 24px; margin-bottom: 32px; display: flex; justify-content: space-between; align-items: center; box-shadow: 0 1px 3px rgba(0,0,0,0.05); }
        .page-title { font-size: 28px; font-weight: 700; color: #0f172a; display: flex; align-items: center; gap: 12px; }
        .help-badge { width: 28px; height: 28px; background: #dbeafe; color: #2563eb; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 14px; font-weight: 700; cursor: help; }
        .user-info { display: flex; align-items: center; gap: 12px; }
        .user-avatar { width: 40px; height: 40px; background: #0066b2; color: white; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-weight: 600; font-size: 14px; }
        .user-details { text-align: right; }
        .user-name { font-weight: 600; font-size: 14px; color: #0f172a; }
        .user-role { font-size: 12px; color: #64748b; }
        
        .info-box { background: #dbeafe; border-left: 4px solid #2563eb; padding: 20px; border-radius: 8px; margin-bottom: 32px; }
        .info-box-title { font-weight: 700; color: #0f172a; margin-bottom: 8px; font-size: 16px; }
        .info-box-content { color: #334155; font-size: 14px; line-height: 1.6; }
        .info-box ul { margin-left: 20px; margin-top: 8px; }
        .info-box li { margin-bottom: 4px; }
        
        .chart-section { background: white; border-radius: 12px; padding: 24px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); margin-bottom: 32px; }
        .chart-title { font-size: 20px; font-weight: 700; color: #0f172a; margin-bottom: 20px; }
        .chart-container { position: relative; height: 400px; }
        
        .table-section { background: white; border-radius: 12px; padding: 24px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); overflow-x: auto; }
        table { width: 100%; border-collapse: collapse; }
        thead { background: #f8fafc; position: sticky; top: 0; }
        th { text-align: left; padding: 12px 16px; font-size: 12px; font-weight: 600; color: #64748b; text-transform: uppercase; letter-spacing: 0.5px; white-space: nowrap; }
        td { padding: 12px 16px; border-bottom: 1px solid #f1f5f9; font-size: 14px; color: #334155; white-space: nowrap; }
        tbody tr:hover { background: #f8fafc; }
        .badge { display: inline-block; padding: 4px 10px; border-radius: 6px; font-size: 12px; font-weight: 600; }
        .badge-success { background: #d1fae5; color: #10b981; }
        .badge-warning { background: #fef3c7; color: #f59e0b; }
        .badge-danger { background: #fee2e2; color: #dc2626; }
        .text-right { text-align: right; }
    </style>
</head>
<body>
    <div class="container">
        <aside class="sidebar">
            <div class="sidebar-header">
                <img src="<?php echo LOGO_URL; ?>" alt="Logo" class="logo">
                <span class="app-name"><?php echo APP_NAME; ?></span>
            </div>
            <ul class="menu">
                <li><a href="../index.php"><span class="menu-icon">📊</span> Dashboard</a></li>
                <li><a href="engineer_performance.php" class="active"><span class="menu-icon">👤</span> Engineer Performance</a></li>
                <li><a href="contract_view.php"><span class="menu-icon">📄</span> Contract Analysis</a></li>
                <li><a href="travel_efficiency.php"><span class="menu-icon">🚗</span> Travel Efficiency</a></li>
                <li><a href="documentation.php"><span class="menu-icon">📝</span> Documentation</a></li>
                <li><a href="../logout.php"><span class="menu-icon">🚪</span> Logout</a></li>
            </ul>
        </aside>
        
        <main class="main-content">
            <div class="topbar">
                <h1 class="page-title">
                    👤 Engineer Performance & Overtime
                    <span class="help-badge" title="View analysis methodology">?</span>
                </h1>
                <div class="user-info">
                    <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['full_name'], 0, 2)); ?></div>
                    <div class="user-details">
                        <div class="user-name"><?php echo $_SESSION['full_name']; ?></div>
                        <div class="user-role"><?php echo ucfirst($_SESSION['role']); ?></div>
                    </div>
                </div>
            </div>
            
            <div class="info-box">
                <div class="info-box-title">📊 Analysis Methodology</div>
                <div class="info-box-content">
                    This analysis provides an overall engineer-level summary. For each engineer, the following metrics are calculated:
                    <ul>
                        <li><strong>Jobs:</strong> Number of unique work orders (WO_NO) handled by the engineer</li>
                        <li><strong>Allocated Hours:</strong> Total and average hours allocated for all jobs</li>
                        <li><strong>Actual Hours:</strong> Total and average actual work time logged</li>
                        <li><strong>Exceeded Jobs:</strong> Count of jobs where actual hours exceeded allocated hours</li>
                        <li><strong>Exceeded Any:</strong> Yes if engineer has any jobs that exceeded allocated hours</li>
                        <li><strong>Overtime Hours:</strong> Total hours worked beyond allocated time (Actual - Allocated, minimum 0)</li>
                        <li><strong>Overtime Ratio:</strong> Overtime hours divided by total allocated hours</li>
                    </ul>
                </div>
            </div>
            
            <?php if (!empty($chartData)): ?>
            <div class="chart-section">
                <div class="chart-title">Top 10 Engineers by Overtime Hours</div>
                <div class="chart-container">
                    <canvas id="overtimeChart"></canvas>
                </div>
            </div>
            
            <div class="chart-section">
                <div class="chart-title">Allocated vs Actual Hours Comparison</div>
                <div class="chart-container">
                    <canvas id="comparisonChart"></canvas>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="table-section">
                <div class="chart-title">Complete Engineer Performance Data</div>
                <table>
                    <thead>
                        <tr>
                            <th>ENGINEER</th>
                            <th class="text-right">JOBS</th>
                            <th class="text-right">TOTAL ALLOCATED</th>
                            <th class="text-right">TOTAL ACTUAL</th>
                            <th class="text-right">AVG ALLOCATED</th>
                            <th class="text-right">AVG ACTUAL</th>
                            <th class="text-right">EXCEEDED JOBS</th>
                            <th>EXCEEDED ANY</th>
                            <th class="text-right">OVERTIME HOURS</th>
                            <th class="text-right">OVERTIME RATIO</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($analysisData as $row): ?>
                        <tr>
                            <td><strong><?php echo htmlspecialchars($row['engineer']); ?></strong></td>
                            <td class="text-right"><?php echo $row['jobs']; ?></td>
                            <td class="text-right"><?php echo formatNumber($row['total_allocated'], 1); ?></td>
                            <td class="text-right"><?php echo formatNumber($row['total_actual'], 1); ?></td>
                            <td class="text-right"><?php echo formatNumber($row['avg_allocated'], 1); ?></td>
                            <td class="text-right"><?php echo formatNumber($row['avg_actual'], 1); ?></td>
                            <td class="text-right">
                                <?php if ($row['exceeded_jobs'] > 0): ?>
                                <span class="badge badge-warning"><?php echo $row['exceeded_jobs']; ?></span>
                                <?php else: ?>
                                <span class="badge badge-success">0</span>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php if ($row['exceeded_any'] === 'Yes'): ?>
                                <span class="badge badge-danger">Yes</span>
                                <?php else: ?>
                                <span class="badge badge-success">No</span>
                                <?php endif; ?>
                            </td>
                            <td class="text-right"><?php echo formatNumber($row['overtime_hours'], 1); ?></td>
                            <td class="text-right"><?php echo formatPercent($row['overtime_ratio']); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </main>
    </div>
    
    <?php if (!empty($chartData)): ?>
    <script>
        // Overtime Chart
        const overtimeCtx = document.getElementById('overtimeChart').getContext('2d');
        new Chart(overtimeCtx, {
            type: 'bar',
            data: {
                labels: <?php echo json_encode($chartData['engineers']); ?>,
                datasets: [{
                    label: 'Overtime Hours',
                    data: <?php echo json_encode($chartData['overtime']); ?>,
                    backgroundColor: 'rgba(245, 158, 11, 0.8)',
                    borderColor: 'rgba(245, 158, 11, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return context.parsed.y.toFixed(1) + ' hours';
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: { display: true, text: 'Hours' }
                    }
                }
            }
        });
        
        // Comparison Chart
        const comparisonCtx = document.getElementById('comparisonChart').getContext('2d');
        new Chart(comparisonCtx, {
            type: 'bar',
            data: {
                labels: <?php echo json_encode($chartData['engineers']); ?>,
                datasets: [
                    {
                        label: 'Allocated Hours',
                        data: <?php echo json_encode($chartData['allocated']); ?>,
                        backgroundColor: 'rgba(37, 99, 235, 0.8)',
                        borderColor: 'rgba(37, 99, 235, 1)',
                        borderWidth: 1
                    },
                    {
                        label: 'Actual Hours',
                        data: <?php echo json_encode($chartData['actual']); ?>,
                        backgroundColor: 'rgba(220, 38, 38, 0.8)',
                        borderColor: 'rgba(220, 38, 38, 1)',
                        borderWidth: 1
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { position: 'top' },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return context.dataset.label + ': ' + context.parsed.y.toFixed(1) + ' hours';
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        title: { display: true, text: 'Hours' }
                    }
                }
            }
        });
    </script>
    <?php endif; ?>
</body>
</html>
