<?php
require_once '../config.php';
require_once '../functions.php';
requireLogin();

$analysisData = [];
$chartData = [];

if (checkDataFile()) {
    $data = readCSVData();
    $engineerData = [];
    
    // Process all rows
    foreach ($data as $row) {
        $engName = $row[16] ?? 'Unknown';
        $docAttached = strtolower(trim($row[26] ?? ''));
        
        if (!isset($engineerData[$engName])) {
            $engineerData[$engName] = [
                'total_jobs' => 0,
                'documented_jobs' => 0
            ];
        }
        
        $engineerData[$engName]['total_jobs']++;
        
        if ($docAttached === 'yes' || $docAttached === 'true' || $docAttached === '1') {
            $engineerData[$engName]['documented_jobs']++;
        }
    }
    
    // Calculate final metrics
    foreach ($engineerData as $name => $data) {
        $docRate = $data['total_jobs'] > 0 ? $data['documented_jobs'] / $data['total_jobs'] : 0;
        
        $analysisData[] = [
            'engineer' => $name,
            'jobs' => $data['total_jobs'],
            'documented' => $data['documented_jobs'],
            'undocumented' => $data['total_jobs'] - $data['documented_jobs'],
            'doc_rate' => $docRate
        ];
    }
    
    // Sort by doc rate ascending (lowest compliance first)
    usort($analysisData, function($a, $b) {
        return $a['doc_rate'] <=> $b['doc_rate'];
    });
    
    // Prepare chart data (bottom 10 for improvement focus)
    $bottom10 = array_slice($analysisData, 0, 10);
    foreach ($bottom10 as $item) {
        $chartData['engineers'][] = $item['engineer'];
        $chartData['doc_rate'][] = round($item['doc_rate'] * 100, 1);
        $chartData['documented'][] = $item['documented'];
        $chartData['undocumented'][] = $item['undocumented'];
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Documentation Compliance - <?php echo APP_NAME; ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Manrope:wght@400;500;600;700&display=swap" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: 'Manrope', sans-serif; background: #f8fafc; color: #1e293b; }
        .container { display: flex; min-height: 100vh; }
        
        .sidebar { width: 260px; background: white; border-right: 1px solid #e2e8f0; padding: 24px 0; position: fixed; height: 100vh; overflow-y: auto; }
        .sidebar-header { padding: 0 24px 24px; border-bottom: 1px solid #e2e8f0; display: flex; align-items: center; gap: 12px; }
        .logo { height: 32px; }
        .app-name { font-size: 20px; font-weight: 700; color: #0066b2; }
        .menu { list-style: none; padding: 16px 0; }
        .menu li { margin-bottom: 4px; }
        .menu a { display: flex; align-items: center; gap: 12px; padding: 12px 24px; color: #64748b; text-decoration: none; font-size: 15px; font-weight: 500; transition: all 0.2s; }
        .menu a:hover, .menu a.active { background: #f1f5f9; color: #0066b2; }
        .menu-icon { width: 20px; height: 20px; display: flex; align-items: center; justify-content: center; }
        
        .main-content { flex: 1; margin-left: 260px; padding: 32px; }
        .topbar { background: white; border-radius: 12px; padding: 16px 24px; margin-bottom: 32px; display: flex; justify-content: space-between; align-items: center; box-shadow: 0 1px 3px rgba(0,0,0,0.05); }
        .page-title { font-size: 28px; font-weight: 700; color: #0f172a; display: flex; align-items: center; gap: 12px; }
        .help-badge { width: 28px; height: 28px; background: #dbeafe; color: #2563eb; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 14px; font-weight: 700; cursor: help; }
        .user-info { display: flex; align-items: center; gap: 12px; }
        .user-avatar { width: 40px; height: 40px; background: #0066b2; color: white; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-weight: 600; font-size: 14px; }
        .user-details { text-align: right; }
        .user-name { font-weight: 600; font-size: 14px; color: #0f172a; }
        .user-role { font-size: 12px; color: #64748b; }
        
        .info-box { background: #dbeafe; border-left: 4px solid #2563eb; padding: 20px; border-radius: 8px; margin-bottom: 32px; }
        .info-box-title { font-weight: 700; color: #0f172a; margin-bottom: 8px; font-size: 16px; }
        .info-box-content { color: #334155; font-size: 14px; line-height: 1.6; }
        .info-box ul { margin-left: 20px; margin-top: 8px; }
        .info-box li { margin-bottom: 4px; }
        
        .chart-section { background: white; border-radius: 12px; padding: 24px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); margin-bottom: 32px; }
        .chart-title { font-size: 20px; font-weight: 700; color: #0f172a; margin-bottom: 20px; }
        .chart-container { position: relative; height: 400px; }
        
        .table-section { background: white; border-radius: 12px; padding: 24px; box-shadow: 0 1px 3px rgba(0,0,0,0.05); overflow-x: auto; }
        table { width: 100%; border-collapse: collapse; }
        thead { background: #f8fafc; position: sticky; top: 0; }
        th { text-align: left; padding: 12px 16px; font-size: 12px; font-weight: 600; color: #64748b; text-transform: uppercase; letter-spacing: 0.5px; white-space: nowrap; }
        td { padding: 12px 16px; border-bottom: 1px solid #f1f5f9; font-size: 14px; color: #334155; white-space: nowrap; }
        tbody tr:hover { background: #f8fafc; }
        .badge { display: inline-block; padding: 4px 10px; border-radius: 6px; font-size: 12px; font-weight: 600; }
        .badge-success { background: #d1fae5; color: #10b981; }
        .badge-warning { background: #fef3c7; color: #f59e0b; }
        .badge-danger { background: #fee2e2; color: #dc2626; }
        .text-right { text-align: right; }
    </style>
</head>
<body>
    <div class="container">
        <aside class="sidebar">
            <div class="sidebar-header">
                <img src="<?php echo LOGO_URL; ?>" alt="Logo" class="logo">
                <span class="app-name"><?php echo APP_NAME; ?></span>
            </div>
            <ul class="menu">
                <li><a href="../index.php"><span class="menu-icon">📊</span> Dashboard</a></li>
                <li><a href="engineer_performance.php"><span class="menu-icon">👤</span> Engineer Performance</a></li>
                <li><a href="contract_view.php"><span class="menu-icon">📄</span> Contract Analysis</a></li>
                <li><a href="travel_efficiency.php"><span class="menu-icon">🚗</span> Travel Efficiency</a></li>
                <li><a href="documentation.php" class="active"><span class="menu-icon">📝</span> Documentation</a></li>
                <li><a href="../logout.php"><span class="menu-icon">🚪</span> Logout</a></li>
            </ul>
        </aside>
        
        <main class="main-content">
            <div class="topbar">
                <h1 class="page-title">
                    📝 Documentation Compliance
                    <span class="help-badge" title="View analysis methodology">?</span>
                </h1>
                <div class="user-info">
                    <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['full_name'], 0, 2)); ?></div>
                    <div class="user-details">
                        <div class="user-name"><?php echo $_SESSION['full_name']; ?></div>
                        <div class="user-role"><?php echo ucfirst($_SESSION['role']); ?></div>
                    </div>
                </div>
            </div>
            
            <div class="info-box">
                <div class="info-box-title">📊 Analysis Methodology</div>
                <div class="info-box-content">
                    This analysis provides engineer-level documentation compliance statistics. For each engineer, the following is calculated:
                    <ul>
                        <li><strong>Jobs:</strong> Total number of work entries/records for this engineer</li>
                        <li><strong>Documented:</strong> Count of jobs where DOC_ATTACHED is marked as 'Yes', 'True', or '1'</li>
                        <li><strong>Undocumented:</strong> Count of jobs without proper documentation attached</li>
                        <li><strong>Doc Rate:</strong> Compliance percentage calculated as: Documented Jobs / Total Jobs</li>
                    </ul>
                    Engineers with low documentation rates may need additional training or process improvements. The table is sorted by lowest doc rates first to highlight areas needing attention.
                </div>
            </div>
            
            <?php if (!empty($chartData)): ?>
            <div class="chart-section">
                <div class="chart-title">Bottom 10 Engineers by Documentation Rate (Need Attention)</div>
                <div class="chart-container">
                    <canvas id="docRateChart"></canvas>
                </div>
            </div>
            
            <div class="chart-section">
                <div class="chart-title">Documented vs Undocumented Jobs</div>
                <div class="chart-container">
                    <canvas id="comparisonChart"></canvas>
                </div>
            </div>
            <?php endif; ?>
            
            <div class="table-section">
                <div class="chart-title">Complete Documentation Compliance Data</div>
                <table>
                    <thead>
                        <tr>
                            <th>ENGINEER</th>
                            <th class="text-right">TOTAL JOBS</th>
                            <th class="text-right">DOCUMENTED</th>
                            <th class="text-right">UNDOCUMENTED</th>
                            <th class="text-right">DOC RATE</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($analysisData as $row): ?>
                        <tr>
                            <td><strong><?php echo htmlspecialchars($row['engineer']); ?></strong></td>
                            <td class="text-right"><?php echo $row['jobs']; ?></td>
                            <td class="text-right">
                                <span class="badge badge-success"><?php echo $row['documented']; ?></span>
                            </td>
                            <td class="text-right">
                                <?php if ($row['undocumented'] > 0): ?>
                                <span class="badge badge-danger"><?php echo $row['undocumented']; ?></span>
                                <?php else: ?>
                                <span class="badge badge-success">0</span>
                                <?php endif; ?>
                            </td>
                            <td class="text-right">
                                <?php if ($row['doc_rate'] >= 0.9): ?>
                                <span class="badge badge-success"><?php echo formatPercent($row['doc_rate']); ?></span>
                                <?php elseif ($row['doc_rate'] >= 0.7): ?>
                                <span class="badge badge-warning"><?php echo formatPercent($row['doc_rate']); ?></span>
                                <?php else: ?>
                                <span class="badge badge-danger"><?php echo formatPercent($row['doc_rate']); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </main>
    </div>
    
    <?php if (!empty($chartData)): ?>
    <script>
        // Documentation Rate Chart
        const docRateCtx = document.getElementById('docRateChart').getContext('2d');
        new Chart(docRateCtx, {
            type: 'bar',
            data: {
                labels: <?php echo json_encode($chartData['engineers']); ?>,
                datasets: [{
                    label: 'Documentation Rate (%)',
                    data: <?php echo json_encode($chartData['doc_rate']); ?>,
                    backgroundColor: function(context) {
                        const value = context.parsed.y;
                        if (value >= 90) return 'rgba(16, 185, 129, 0.8)';
                        if (value >= 70) return 'rgba(245, 158, 11, 0.8)';
                        return 'rgba(220, 38, 38, 0.8)';
                    },
                    borderColor: function(context) {
                        const value = context.parsed.y;
                        if (value >= 90) return 'rgba(16, 185, 129, 1)';
                        if (value >= 70) return 'rgba(245, 158, 11, 1)';
                        return 'rgba(220, 38, 38, 1)';
                    },
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { display: false },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return context.parsed.y.toFixed(1) + '%';
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        max: 100,
                        title: { display: true, text: 'Documentation Rate (%)' }
                    }
                }
            }
        });
        
        // Documented vs Undocumented Chart
        const comparisonCtx = document.getElementById('comparisonChart').getContext('2d');
        new Chart(comparisonCtx, {
            type: 'bar',
            data: {
                labels: <?php echo json_encode($chartData['engineers']); ?>,
                datasets: [
                    {
                        label: 'Documented',
                        data: <?php echo json_encode($chartData['documented']); ?>,
                        backgroundColor: 'rgba(16, 185, 129, 0.8)',
                        borderColor: 'rgba(16, 185, 129, 1)',
                        borderWidth: 1
                    },
                    {
                        label: 'Undocumented',
                        data: <?php echo json_encode($chartData['undocumented']); ?>,
                        backgroundColor: 'rgba(220, 38, 38, 0.8)',
                        borderColor: 'rgba(220, 38, 38, 1)',
                        borderWidth: 1
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: { position: 'top' },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                return context.dataset.label + ': ' + context.parsed.y + ' jobs';
                            }
                        }
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        stacked: false,
                        title: { display: true, text: 'Number of Jobs' }
                    }
                }
            }
        });
    </script>
    <?php endif; ?>
</body>
</html>
